using System;
using System.Windows.Forms;

namespace ScoreBoard
{
	// THIS USER DEFINED TYPE IS USED TO TELL WHICH EVENT HAS OCCURED
	public enum RequestType 
	{
		ChangeInstructionStatus,
		ChangeFunctionalUnitStatus,
		ChangeRegisterStatus,
		ChangeClockCycle
	};
	
	// THIS CLASS WILL SET OR GIVE THE REQUESTED EVENT THAT IS GENERATED
	public class RequestEventArgs : EventArgs
	{
		private RequestType request;

		public RequestEventArgs(RequestType request) : base()
		{
			this.request = request;
		}

		public RequestType Request
		{
			get
			{
				return request;
			}
		}
	}

	/// <summary>
	/// SCHEDULING CLASS :	THIS IS VERY INPORTANT CLASS WHICH WILL HANDLE ALL THE
	///						SCHEDULING OF THE INSTRCUTIONS AS THE INSTRUCTIONS ARE
	///						EXECUTED OUT-OF-ORDER. THIS CLASS ALSO HANDLES THE
	///						HAZARDS (I.E. WAW , WAR ETC)
	/// </summary>
	public class Scheduler
	{
		private ScoreBoard refScoreBoard;			// REFERENCE OF THE MAIN CLASS I.E.
													// SCOREBOARD (2 WAY COMMUNICATION)

		public delegate void UpdateRequest(object sender, RequestEventArgs e);
		public event UpdateRequest OnUpdateRequest;
			
		// ARGUMENTED CONSTRUCTOR
		public Scheduler(ScoreBoard refSB)
		{
			refScoreBoard = refSB;
		}

		// THIS IS THE MAIN FUCNTION , THAT HANDLES THE COMPLETE PROJECT
		// DURING THIS SINGLE CYCLE ALL THE THINGS ARE DONE UPTO THE EXECPTION
		public bool UpdateCycle()
		{
			bool hasSchedDone = true;
			// check if scheduling complete
			foreach (Instruction ins in refScoreBoard.InstList)
			{
				// CHECK WHETHER THE INSTRUCTION IS IN THE LAST STAGE (I.E. WRITE RESULT)
				if ( !(ins.currentStage == Stage.WriteResult &&
					ins.hasCurrentStageCompleted) )
				{
					hasSchedDone = false;
					break;
				}
			}

			// IF THE SCHEDULING IS DONE, IT CLEARS THE FUNCTIONAL UNITS
			if( hasSchedDone )
			{
				for(int i=0; i<refScoreBoard.FunctionalUnit.Count; i++)
				{
					((FU)(refScoreBoard.FunctionalUnit[i])).Flush();
				}
				return false;
			}
			
			// INCREMENT THE CLOCK TICK
			refScoreBoard.clock.Increment();

			// UPDATE THE INSTRUCTION AND THE REGISTER UNIT
			UpdateComponents();

			bool isAnyIssued = false;

			// CHECK FOR THE STRUCTUAL HAZARD
			for( int i=0; i<refScoreBoard.InstList.Count; i++ )
			{
				Instruction ins = (Instruction)(refScoreBoard.InstList[i]);
				
				// if instruction wants to write back results
				#region WriteResult	
				if( ins.currentStage == Stage.WriteResult 
					&& !(ins.hasCurrentStageCompleted) )
				{
					if( TryToWriteResult(ins) )
					{
						ins.WriteResult = refScoreBoard.clock.Counter;
						ins.hasCurrentStageCompleted = true;
						ins.OccupiedFU.shouldBeFreed = true;
						SendUpdateEvent();
					}
					continue; 
				}
				#endregion

				// if instruction is in execution stage
				#region Execute 
				else if( ins.currentStage == Stage.Execute )
				{
					ins.ExecutionCounter--;
					if(ins.ExecutionCounter == 0)
					{
						ins.ExeComp = refScoreBoard.clock.Counter;
						ins.currentStage = Stage.WriteResult;
					}
					SendUpdateEvent();
					continue;
				}
				#endregion

				// if the instruction has been issued and waiting
				// to read operands
				#region Waiting To Read Operands
				else if( ins.currentStage == Stage.ReadOperands )
				{
					if( TryToReadOperands(ins) )
					{
						ins.ReadOP = refScoreBoard.clock.Counter;
						ins.currentStage = Stage.Execute;
						ins.hasCurrentStageCompleted = false;
						SendUpdateEvent();
						continue; 
					}
				}
				#endregion

				// if the instruction has not been issued
				#region Waiting To Be Issued
				else if( ins.currentStage == Stage.Issue && !(ins.hasCurrentStageCompleted)
					&& !isAnyIssued)
				{
					if( TryToIssue(ins) )
					{
						isAnyIssued = true;
						ins.currentStage = Stage.ReadOperands;
						ins.hasCurrentStageCompleted = false;
						ins.Issue = refScoreBoard.clock.Counter;
						ins.OccupiedFU.Busy = true;
						ins.OccupiedFU.Op = ins.OPCode;
						ins.OccupiedFU.Fi = ins.Dest;
						ins.OccupiedFU.Fj = ins.Src1;
						ins.OccupiedFU.Fk = ins.Src2;
						ins.OccupiedFU.inst = ins;
						SendUpdateEvent();
					}
					else
						break;
				}
				#endregion
			}
			return true;
		}

		// UPDATE THE INSTRUCTION AND THE REGISTER UNIT AND THE SCOREBOARDING TABLE AS WELL
		public void UpdateComponents()
		{
			for( int i=0; i<refScoreBoard.FunctionalUnit.Count; i++ )
			{
				FU fu = ((FU)refScoreBoard.FunctionalUnit[i]);
				if( fu.shouldBeFreed ) 
				{					
					if( fu.Fi[0] == 'F' )
					{
						string dest = fu.Fi.Substring(1);
						int idx = Convert.ToInt32(dest);
						FR destreg = ((FR)(refScoreBoard.RegisterSet[idx]));
						destreg.Reg = "";
					}
					fu.Flush();
				}
			}

			// check for source register availability
			for( int i=0; i<refScoreBoard.FunctionalUnit.Count; i++ )
			{
				FU fu = ((FU)refScoreBoard.FunctionalUnit[i]);
				if( fu.Busy )
				{
					if( fu.Fj != "" && !(fu.Rj) )
					{
						string src = fu.Fj.Substring(1);
						int idx = Convert.ToInt32(src);
						FR srcreg = ((FR)(refScoreBoard.RegisterSet[idx]));
						if( srcreg.Reg == "" )
							fu.Rj = true;
					}

					if( fu.Fk != "" && !(fu.Rk) )
					{
						string src = fu.Fk.Substring(1);
						int idx = Convert.ToInt32(src);
						FR srcreg = ((FR)(refScoreBoard.RegisterSet[idx]));
						if( srcreg.Reg == "" )
							fu.Rk = true;
					}
				}
			}

			// update registers
			for( int i=0; i<refScoreBoard.RegisterSet.Count; i++ )
			{
				((FR)refScoreBoard.RegisterSet[i]).JustRead = false; 
			}
		
		}

		public bool TryToIssue(Instruction inst)
		{
			FU fu = null;

			string dest = inst.Dest.Substring(1);
			int idx = Convert.ToInt32(dest);
			FR destreg = ((FR)(refScoreBoard.RegisterSet[idx]));

			if(inst.OPCode == "LD")
			{
				fu = ((FU)refScoreBoard.FunctionalUnit[0]);
								
				if( fu.Busy )
					return false;

				destreg.Reg = fu.Name;
				inst.OccupiedFU = fu;
			}
			
			else if(inst.OPCode == "MUL")
			{
				if( ((FU)refScoreBoard.FunctionalUnit[1]).Busy
				  && ((FU)refScoreBoard.FunctionalUnit[2]).Busy )
					return false;
					
				fu = (((FU)refScoreBoard.FunctionalUnit[1]).Busy) 
					? ((FU)refScoreBoard.FunctionalUnit[2]) : 
					  ((FU)refScoreBoard.FunctionalUnit[1]);
				
				inst.OccupiedFU =fu;
				destreg.Reg = fu.Name;		
			}
					
			else if(inst.OPCode == "ADD" || inst.OPCode == "SUB")
			{
				fu = ((FU)refScoreBoard.FunctionalUnit[3]);

				if( fu.Busy )
					return false;

				inst.OccupiedFU = fu;
				destreg.Reg = "Add";
			}
		
			else if(inst.OPCode == "DIV")
			{
				fu = ((FU)refScoreBoard.FunctionalUnit[4]);

				if( fu.Busy )
					return false;

				destreg.Reg = fu.Name;
				inst.OccupiedFU = fu;
			}

			// check if source registers are available
			string src = inst.Src1;
			if(src[0] == 'F')
			{
				src = src.Substring(1);
				int index = Convert.ToInt32(src);
				FR reg = ((FR)(refScoreBoard.RegisterSet[index]));
				inst.OccupiedFU.Rj = (reg.Reg == "");
			}

			src = inst.Src2;
			if(src[0] == 'F')
			{
				src = src.Substring(1);
				int index = Convert.ToInt32(src);
				FR reg = ((FR)(refScoreBoard.RegisterSet[index]));
				inst.OccupiedFU.Rk = (reg.Reg == "");
			}

			return true;
		}

		public void SendUpdateEvent()
		{
			OnUpdateRequest(this, 
				new RequestEventArgs(RequestType.ChangeFunctionalUnitStatus));
			OnUpdateRequest(this, 
				new RequestEventArgs(RequestType.ChangeInstructionStatus));
			OnUpdateRequest(this, 
				new RequestEventArgs(RequestType.ChangeRegisterStatus));
		}

		public bool TryToReadOperands(Instruction inst)
		{
			if( inst.OccupiedFU.Rj && inst.OccupiedFU.Rk )
			{
				// Mark registers as just read
				string src = inst.Src1;
				if(src[0] == 'F')
				{
					src = src.Substring(1);
					int index = Convert.ToInt32(src);
					FR reg = ((FR)(refScoreBoard.RegisterSet[index]));
					reg.JustRead = true;
				}

				src = inst.Src2;
				if(src[0] == 'F')
				{
					src = src.Substring(1);
					int index = Convert.ToInt32(src);
					FR reg = ((FR)(refScoreBoard.RegisterSet[index]));
					reg.JustRead = true;
				}
				return true;
			}
			return false;
		}

		public bool TryToWriteResult(Instruction inst)
		{
			// Check if registers have just been read in the
			// current cycle for WAR hazard
			string dest = inst.Dest;
			if(dest[0] == 'F')
			{
				dest = dest.Substring(1);
				int index = Convert.ToInt32(dest);
				FR reg = ((FR)(refScoreBoard.RegisterSet[index]));
				if(reg.JustRead)
					return false;
			}

			// check all previous instructions for WAR hazard
			for( int i=0; i<inst.InstructionPosition; i++ )
			{
				Instruction prevIns = (Instruction)(refScoreBoard.InstList[i]);
				
				// check if instruction is waiting to read
				if( prevIns.currentStage == Stage.ReadOperands )
				{
					if( (prevIns.Src1 == inst.Dest) || 
						(prevIns.Src2 == inst.Dest) )
						return false;
				}
			}
			return true;
		}
	}
}
